import { useState, useEffect } from 'react'
import { motion } from 'framer-motion'
import { Box, User, Calendar, Star, Code, Zap, Award, TrendingUp, Layers, Target } from 'lucide-react'
import ImageSlider from '../components/ImageSlider'
import './MyProjects.css'

const MyProjects = () => {
  const [projects, setProjects] = useState([])

  useEffect(() => {
    const saved = localStorage.getItem('model3dProjects')
    if (saved) {
      try {
        setProjects(JSON.parse(saved))
      } catch (e) {
        setProjects([])
      }
    } else {
      const defaultProjects = [
        {
          id: 1,
          course: 'Blender для начинающих',
          instructor: 'Алексей Волков',
          submissionDate: '2025-03-28',
          rating: 5,
          progress: 85
        },
        {
          id: 2,
          course: 'Maya Advanced',
          instructor: 'Мария Соколова',
          submissionDate: '2025-03-25',
          rating: 5,
          progress: 92
        },
        {
          id: 3,
          course: '3ds Max Профессионал',
          instructor: 'Дмитрий Новиков',
          submissionDate: '2025-03-22',
          rating: 4,
          progress: 78
        }
      ]
      setProjects(defaultProjects)
      localStorage.setItem('model3dProjects', JSON.stringify(defaultProjects))
    }
  }, [])

  const totalProjects = projects.length
  const averageRating = projects.length > 0
    ? (projects.reduce((sum, p) => sum + p.rating, 0) / projects.length).toFixed(1)
    : 0
  const totalProgress = projects.length > 0
    ? Math.round(projects.reduce((sum, p) => sum + p.progress, 0) / projects.length)
    : 0
  const completedProjects = projects.filter(p => p.progress === 100).length

  return (
    <div className="my-projects">
      <div className="container">
        <motion.section 
          className="hero-tech"
          initial={{ opacity: 0, y: 40 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.8 }}
        >
          <div className="hero-tech-grid">
            <motion.div 
              className="hero-tech-main"
              initial={{ opacity: 0, x: -60 }}
              animate={{ opacity: 1, x: 0 }}
              transition={{ duration: 0.8, delay: 0.2 }}
            >
              <div className="hero-tech-badge">
                <Zap size={20} strokeWidth={2.5} />
                <span>3D МОДЕЛИРОВАНИЕ</span>
              </div>
              <h1 className="hero-tech-title">
                Мои <span className="hero-tech-accent">проекты</span>
              </h1>
              <div className="hero-tech-line"></div>
            </motion.div>
            <motion.div 
              className="hero-tech-side"
              initial={{ opacity: 0, x: 60 }}
              animate={{ opacity: 1, x: 0 }}
              transition={{ duration: 0.8, delay: 0.3 }}
            >
              <p className="hero-tech-description">
                Отслеживайте прогресс ваших 3D-проектов и достижения в обучении
              </p>
              <div className="hero-tech-stats">
                <div className="hero-tech-stat">
                  <Code size={20} strokeWidth={2.5} />
                  <span>{totalProjects} проектов</span>
                </div>
                <div className="hero-tech-stat">
                  <Star size={20} strokeWidth={2.5} />
                  <span>{averageRating} рейтинг</span>
                </div>
              </div>
            </motion.div>
          </div>
        </motion.section>

        <motion.section 
          className="stats-tech"
          initial={{ opacity: 0, y: 40 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.7, delay: 0.4 }}
        >
          <div className="stats-tech-grid">
            <motion.div 
              className="stat-tech-card"
              whileHover={{ scale: 1.05, y: -8, transition: { duration: 0.3 } }}
            >
              <div className="stat-tech-icon" style={{ background: 'linear-gradient(135deg, #00D9FF 0%, #7C3AED 100%)' }}>
                <Box size={40} strokeWidth={2.5} color="white" />
                <div className="stat-tech-icon-glow"></div>
              </div>
              <div className="stat-tech-content">
                <div className="stat-tech-value">{totalProjects}</div>
                <div className="stat-tech-label">Проектов</div>
              </div>
              <div className="stat-tech-corner"></div>
            </motion.div>
            <motion.div 
              className="stat-tech-card"
              whileHover={{ scale: 1.05, y: -8, transition: { duration: 0.3 } }}
            >
              <div className="stat-tech-icon" style={{ background: 'linear-gradient(135deg, #7C3AED 0%, #F59E0B 100%)' }}>
                <Star size={40} strokeWidth={2.5} color="white" />
                <div className="stat-tech-icon-glow"></div>
              </div>
              <div className="stat-tech-content">
                <div className="stat-tech-value">{averageRating}</div>
                <div className="stat-tech-label">Рейтинг</div>
              </div>
              <div className="stat-tech-corner"></div>
            </motion.div>
            <motion.div 
              className="stat-tech-card"
              whileHover={{ scale: 1.05, y: -8, transition: { duration: 0.3 } }}
            >
              <div className="stat-tech-icon" style={{ background: 'linear-gradient(135deg, #00D9FF 0%, #F59E0B 100%)' }}>
                <TrendingUp size={40} strokeWidth={2.5} color="white" />
                <div className="stat-tech-icon-glow"></div>
              </div>
              <div className="stat-tech-content">
                <div className="stat-tech-value">{totalProgress}%</div>
                <div className="stat-tech-label">Прогресс</div>
              </div>
              <div className="stat-tech-corner"></div>
            </motion.div>
            <motion.div 
              className="stat-tech-card"
              whileHover={{ scale: 1.05, y: -8, transition: { duration: 0.3 } }}
            >
              <div className="stat-tech-icon" style={{ background: 'linear-gradient(135deg, #F59E0B 0%, #7C3AED 100%)' }}>
                <Award size={40} strokeWidth={2.5} color="white" />
                <div className="stat-tech-icon-glow"></div>
              </div>
              <div className="stat-tech-content">
                <div className="stat-tech-value">{completedProjects}</div>
                <div className="stat-tech-label">Завершено</div>
              </div>
              <div className="stat-tech-corner"></div>
            </motion.div>
          </div>
        </motion.section>

        <ImageSlider images={[
          '/images/model-1.jpg',
          '/images/model-2.jpg',
          '/images/model-3.jpg',
          '/images/model-4.jpg'
        ]} />

        <motion.section 
          className="projects-tech"
          initial={{ opacity: 0, y: 40 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.7, delay: 0.5 }}
        >
          <div className="section-tech-header">
            <div className="section-tech-line"></div>
            <h2 className="section-tech-title">Мои проекты</h2>
            <div className="section-tech-accent"></div>
          </div>
          {projects.length > 0 ? (
            <div className="projects-tech-grid">
              {projects.map((project, index) => (
                <motion.div
                  key={project.id}
                  className="project-tech-card"
                  initial={{ opacity: 0, y: 40, rotateX: -15 }}
                  animate={{ opacity: 1, y: 0, rotateX: 0 }}
                  transition={{ delay: 0.6 + index * 0.1, duration: 0.6 }}
                  whileHover={{ 
                    y: -12, 
                    rotateY: 5,
                    scale: 1.02,
                    transition: { duration: 0.3 } 
                  }}
                >
                  <div className="project-tech-header">
                    <div className="project-tech-badge" style={{ 
                      background: index % 3 === 0 ? 'linear-gradient(135deg, #00D9FF 0%, #7C3AED 100%)' :
                      index % 3 === 1 ? 'linear-gradient(135deg, #7C3AED 0%, #F59E0B 100%)' :
                      'linear-gradient(135deg, #00D9FF 0%, #F59E0B 100%)'
                    }}>
                      {project.course.split(' ')[0]}
                    </div>
                    <div className="project-tech-rating">
                      {Array.from({ length: 5 }, (_, i) => (
                        <Star
                          key={i}
                          size={18}
                          fill={i < project.rating ? '#F59E0B' : 'none'}
                          stroke={i < project.rating ? '#F59E0B' : 'rgba(0, 217, 255, 0.3)'}
                          strokeWidth={2.5}
                        />
                      ))}
                    </div>
                  </div>
                  <h3 className="project-tech-course">{project.course}</h3>
                  <div className="project-tech-details">
                    <div className="project-tech-detail">
                      <User size={18} strokeWidth={2.5} color="#00D9FF" />
                      <span>{project.instructor}</span>
                    </div>
                    <div className="project-tech-detail">
                      <Calendar size={18} strokeWidth={2.5} color="#7C3AED" />
                      <span>{new Date(project.submissionDate).toLocaleDateString('ru-RU', { day: 'numeric', month: 'long', year: 'numeric' })}</span>
                    </div>
                  </div>
                  <div className="project-tech-progress">
                    <div className="project-tech-progress-header">
                      <span className="progress-label">Прогресс</span>
                      <span className="progress-value">{project.progress}%</span>
                    </div>
                    <div className="project-tech-progress-bar">
                      <motion.div 
                        className="project-tech-progress-fill"
                        initial={{ width: 0 }}
                        animate={{ width: `${project.progress}%` }}
                        transition={{ delay: 0.8 + index * 0.1, duration: 1, ease: "easeOut" }}
                        style={{ 
                          background: index % 3 === 0 ? 'linear-gradient(90deg, #00D9FF 0%, #7C3AED 100%)' :
                          index % 3 === 1 ? 'linear-gradient(90deg, #7C3AED 0%, #F59E0B 100%)' :
                          'linear-gradient(90deg, #00D9FF 0%, #F59E0B 100%)'
                        }}
                      />
                    </div>
                  </div>
                  <div className="project-tech-glow"></div>
                </motion.div>
              ))}
            </div>
          ) : (
            <div className="empty-tech">
              <div className="empty-tech-icon">
                <Box size={80} strokeWidth={1.5} />
                <div className="empty-tech-rings"></div>
              </div>
              <h3 className="empty-tech-title">Начните свой путь в 3D</h3>
              <p className="empty-tech-text">Запишитесь на курс и увидите свой первый проект здесь</p>
            </div>
          )}
        </motion.section>

        <motion.section 
          className="features-tech"
          initial={{ opacity: 0, y: 40 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.7, delay: 0.7 }}
        >
          <div className="features-tech-header">
            <div className="section-tech-line"></div>
            <h2 className="section-tech-title">Преимущества обучения</h2>
          </div>
          <div className="features-tech-grid">
            {[
              { icon: Layers, title: 'Структурированная программа', text: 'Пошаговое обучение от основ до продвинутых техник', gradient: 'linear-gradient(135deg, #00D9FF 0%, #7C3AED 100%)' },
              { icon: Target, title: 'Практические проекты', text: 'Реальные задачи и портфолио для вашей карьеры', gradient: 'linear-gradient(135deg, #7C3AED 0%, #F59E0B 100%)' },
              { icon: Zap, title: 'Обратная связь', text: 'Персональные консультации с опытными преподавателями', gradient: 'linear-gradient(135deg, #00D9FF 0%, #F59E0B 100%)' },
              { icon: Award, title: 'Сертификаты', text: 'Официальные документы о прохождении курсов', gradient: 'linear-gradient(135deg, #F59E0B 0%, #7C3AED 100%)' }
            ].map((feature, index) => {
              const IconComponent = feature.icon
              return (
                <motion.div
                  key={feature.title}
                  className="feature-tech-card"
                  initial={{ opacity: 0, y: 40, rotateX: -15 }}
                  animate={{ opacity: 1, y: 0, rotateX: 0 }}
                  transition={{ delay: 0.8 + index * 0.1, duration: 0.6 }}
                  whileHover={{ 
                    y: -10, 
                    rotateY: 5,
                    scale: 1.03,
                    transition: { duration: 0.3 } 
                  }}
                >
                  <div className="feature-tech-icon" style={{ background: feature.gradient }}>
                    <IconComponent size={44} strokeWidth={2.5} color="white" />
                    <div className="feature-tech-icon-glow"></div>
                  </div>
                  <div className="feature-tech-content">
                    <h3 className="feature-tech-title">{feature.title}</h3>
                    <p className="feature-tech-text">{feature.text}</p>
                  </div>
                  <div className="feature-tech-border"></div>
                </motion.div>
              )
            })}
          </div>
        </motion.section>
      </div>
    </div>
  )
}

export default MyProjects

